<?php

//=============================================================================
//= MENU CALLBACKS
//=============================================================================

/**
 * Successful payment callback.
 */
function uc_gopay_payment_success() {
  uc_gopay_watchlog('uc_gopay', print_r(dbtlog(), 1));
  _uc_gopay_check_response_access();
  
  if (uc_gopay_setting('debug')) {
    watchdog('uc_gopay', 'Access to payment success page. $_REQUEST = %request', array(
      '%request' => print_r($_REQUEST, TRUE)
    ), WATCHDOG_DEBUG);
  }
  
  try {
    $lock_name = 'uc_gopay_' . $_GET['paymentSessionId'];
    
    if (!lock_acquire($lock_name, 5)) {
      lock_wait($lock_name);
      lock_acquire($lock_name, 5);
    }
    
    _uc_gopay_complete();
    
    lock_release($lock_name);
    
    // This lets us know it's a legitimate access of the complete page.
    $_SESSION['do_complete'] = TRUE;
    drupal_goto('cart/checkout/complete');
  }
  catch (GopayPaymentException $e) {
    drupal_set_message(t('Error during processing the payment: %error', array(
      '%error' => $e->getMessage()
    )), 'error');
    drupal_goto('cart/checkout/review');
  }
}

/**
 * Failed payment callback.
 */
function uc_gopay_payment_failed() {
  uc_gopay_watchlog('uc_gopay', print_r(dbtlog(), 1));
  // Available in $_GET: paymentSessionId, orderNumber, encryptedSignature, targetGoId
  if (empty($_GET['targetGoId'])) {
    drupal_access_denied();
    exit;
  }
  if (uc_gopay_setting('debug')) {
    watchdog('uc_gopay', 'Access to payment failed page. $_REQUEST = %request', array(
      '%request' => print_r($_REQUEST, TRUE)
    ), WATCHDOG_DEBUG);
  }
  unset($_SESSION['cart_order']);
  drupal_set_message(t('Your GoPay payment was canceled. Please feel free to continue shopping or contact us for assistance.'));
  drupal_goto('cart');
}

/**
 * Notify callback.
 */
function uc_gopay_soap_notify() {
  uc_gopay_watchlog('uc_gopay', print_r(dbtlog(), 1));
  
  _uc_gopay_check_response_access();
  
  if (uc_gopay_setting('debug')) {
    watchdog('uc_gopay', 'Access to payment notify page. $_REQUEST: <pre>@request</pre>', array(
      '@request' => print_r($_REQUEST, TRUE)
    ), WATCHDOG_DEBUG);
  }
  
  try {
    $lock_name = 'uc_gopay_' . $_GET['paymentSessionId'];
    
    if (!lock_acquire($lock_name, 5)) {
      lock_wait($lock_name);
      lock_acquire($lock_name, 5);
    }
    
    _uc_gopay_complete();
    
    lock_release($lock_name);
  }
  catch (GopayPaymentException $e) {
    _uc_gopay_server_error();
  }
}

/**
 * Admin form of the GoPay methods.
 */
function uc_gopay_payment_methods_list() {
  uc_gopay_watchlog('uc_gopay', print_r(dbtlog(), 1));
  
  $form = $options = array();
  
  $form['uc_gopay_back_top'] = array(
    '#prefix' => '<p>',
    '#value' => l(t('Back to GoPay settings'), 'admin/store/settings/payment/method/gopay_wps/'),
    '#suffix' => '</p>'
  );
  
  $header              = array(
    t('Logo'),
    t('Payment method'),
    t('Code'),
    t('Offline')
  );
  $payment_method_list = uc_gopay_channel_list_all();
  
  if (module_exists('elements')) {
    foreach ($payment_method_list as &$payment_method) {
      $row                            = array();
      $row[]                          = '<img src="' . $payment_method->logo . '">';
      $row[]                          = $payment_method->paymentMethod;
      $row[]                          = $payment_method->code;
      $row[]                          = ($payment_method->offline == 1) ? t('yes') : t('no');
      $options[$payment_method->code] = $row;
    }
    
    $form['uc_gopay_enabled_channels'] = array(
      '#type' => 'tableselect',
      '#header' => $header,
      '#options' => $options,
      '#empty' => t('There are not payment channels available. Server connection error?'),
      '#default_value' => uc_gopay_setting('channels')
    );
  } else {
    foreach ($payment_method_list as &$payment_method) {
      $row                            = sprintf('<img src="%s"> <b>%s</b> <code>(code: %s, offline: %s)</code>', $payment_method->logo, $payment_method->paymentMethod, $payment_method->code, ($payment_method->offline == 1) ? t('yes') : t('no'));
      $options[$payment_method->code] = $row;
    }
    
    if ($options) {
      $form['uc_gopay_enabled_channels'] = array(
        '#type' => 'checkboxes',
        '#options' => $options,
        '#default_value' => uc_gopay_setting('channels')
      );
    } else {
      $form['uc_gopay_enabled_channels_empty'] = array(
        '#type' => 'markup',
        '#prefix' => '<p>',
        '#suffix' => '</p>',
        '#value' => t('There are not payment channels available. Server connection error?')
      );
    }
  }
  
  $form['uc_gopay_back_bottom'] = array(
    '#prefix' => '<p>',
    '#value' => l(t('Back to GoPay settings'), 'admin/store/settings/payment/method/gopay_wps/'),
    '#suffix' => '</p>'
  );
  
  return system_settings_form($form);
}


//=============================================================================
//= HELPERS
//=============================================================================

class GopayPaymentException extends Exception {
}

/**
 * Returns 200 OK HTTP header.
 */
function _uc_gopay_server_success() {
  uc_gopay_watchlog('uc_gopay', print_r(dbtlog(), 1));
  if (uc_gopay_setting('debug')) {
    watchdog('uc_gopay', 'Server SUCCESS.');
  }
  header('HTTP/1.1 200 OK');
  exit(0);
}

/**
 * Returns 500 Server error HTTP header.
 */
function _uc_gopay_server_error() {
  uc_gopay_watchlog('uc_gopay', print_r(dbtlog(), 1));
  if (uc_gopay_setting('debug')) {
    watchdog('uc_gopay', 'Server ERROR.');
  }
  header('HTTP/1.1 500 Internal Server Error');
  exit(0);
}

/**
 * Splits variable symbol which came back from GoPay to order_id and cart_id.
 *
 * @param string $variable_symbol
 * @return array of order_id [0] and cart_id [1]
 */
function _uc_gopay_parse_variable_symbol($variable_symbol) {
  uc_gopay_watchlog('uc_gopay', print_r(dbtlog(), 1));
  // Parse Order ID and Cart ID
  @list($order_id, $cart_id) = explode('-', $variable_symbol);
  // Sanitize order ID and cart ID
  $order_id = intval($order_id);
  $cart_id  = $cart_id ? check_plain($cart_id) : 0;
  return array(
    $order_id,
    $cart_id
  );
}

/**
 * Checks if paymentSessionId, orderNumber, encryptedSignature, targetGoId are
 * available in $_GET.
 */
function _uc_gopay_check_response_access() {
  uc_gopay_watchlog('uc_gopay', print_r(dbtlog(), 1));
  // Available in $_GET: paymentSessionId, orderNumber, encryptedSignature, targetGoId
  if (empty($_GET['targetGoId']) || empty($_GET['paymentSessionId']) || empty($_GET['orderNumber']) || empty($_GET['encryptedSignature'])) {
    drupal_access_denied();
    exit(1);
  }
}

/**
 * Process the notification returned from the GoPay module to finish the payment
 * process.
 *
 * @throws GopayPaymentException
 */
function _uc_gopay_complete() {
  uc_gopay_watchlog('uc_gopay', print_r(dbtlog(), 1));
  // Assing the values from the GoPay request
  $returned_variable_symbol     = $_GET['orderNumber'];
  $returned_payment_session_id  = $_GET['paymentSessionId'];
  $returned_goid                = $_GET['targetGoId'];
  $returned_encrypted_signature = $_GET['encryptedSignature'];
  
  // Parse Order ID and Cart ID
  list($order_id, $cart_id) = _uc_gopay_parse_variable_symbol($returned_variable_symbol);
  
  if (!empty($cart_id)) {
    // Needed later by uc_complete_sale to empty the correct cart
    $_SESSION['uc_cart_id'] = $cart_id;
  }
  
  watchdog('uc_gopay', 'Receiving PN at URL for order @order_id. <pre>@debug</pre>', array(
    '@order_id' => $order_id,
    '@debug' => uc_gopay_setting('debug') ? print_r($_GET, TRUE) : ''
  ), WATCHDOG_NOTICE, l(t('view'), 'admin/store/orders/' . $order_id));
  
  $payment = uc_gopay_record_find_by_session_id($returned_payment_session_id);
  if (uc_gopay_setting('debug')) {
    watchdog('uc_gopay', 'Payment session record. <pre>@debug</pre>', array(
      '@debug' => print_r($payment, TRUE)
    ), WATCHDOG_DEBUG, l(t('view'), 'admin/store/orders/' . $order_id));
  }
  
  // TODO: load using PaymentSessionId?
  $order = uc_order_load($order_id);
  if ($order == FALSE) {
    watchdog('uc_gopay', 'GoPay response attempted for non-existent order @order_id.', array(
      '@order_id' => $order_id
    ), WATCHDOG_ERROR, l(t('view'), 'admin/store/orders/' . $order_id));
    throw new GopayPaymentException('GoPay transaction failed verification');
  } elseif (($order->payment_method != UC_GOPAY_STANDARD_PAYMENT_METHOD) && (!defined('UC_GOPAY_INLINE_STANDARD_PAYMENT_METHOD') || ($order->payment_method != UC_GOPAY_INLINE_STANDARD_PAYMENT_METHOD))) {
    watchdog('uc_gopay', 'Attempt to pay order @order_id using GoPay, but pament method %payment_method was selected.', array(
      '@order_id' => $order_id,
      '%payment_method' => $order->payment_method
    ), WATCHDOG_ERROR, l(t('view'), 'admin/store/orders/' . $order_id));
    throw new GopayPaymentException("Missing GoPay payment method in the order");
  }
  
  if (!defined('UC_GOPAY_INLINE_STANDARD_PAYMENT_METHOD') || ($order->payment_method != UC_GOPAY_INLINE_STANDARD_PAYMENT_METHOD)) {
    $eshop_goid       = uc_gopay_setting('goid');
    $eshop_secret     = uc_gopay_setting('secret');
  }
  else {
    $eshop_goid       = uc_gopay_inline_setting('goid');
    $eshop_secret     = uc_gopay_inline_setting('secret');
  }

  $payment_currency = $order->currency;
  
  // Make sure that the notification was send by the GoPay and signed.
  if (!GopayHelper::checkPaymentIdentity((float) $returned_goid, (float) $returned_payment_session_id, $returned_variable_symbol, $returned_encrypted_signature, (float) $eshop_goid, $payment['variable_symbol'], $eshop_secret)) {
    // Not valid notification verification - probably malicius attempt to set order as paid.
    watchdog('uc_gopay', 'GoPay transaction failed verification.', array(), WATCHDOG_ERROR, l(t('view'), 'admin/store/orders/' . $order_id));
    uc_order_comment_save($order_id, 0, t('An GoPay transaction failed verification for this order.'), 'admin');
    throw new GopayPaymentException('GoPay transaction failed verification.');
  }
  
  watchdog('uc_gopay', 'GoPay: transaction @id for @order_id verified.', array(
    '@id' => $returned_payment_session_id,
    '@order_id' => $order_id
  ), WATCHDOG_NOTICE, l(t('view'), 'admin/store/orders/' . $order_id));
  
  // Check if the order is payed
  $result = GopaySoap::isEshopPaymentDone((float) $returned_payment_session_id, (float) $eshop_goid, $payment['variable_symbol'], $payment['total_price'], $payment['product_name'], $eshop_secret);
  if (empty($result['sessionState'])) {
    throw new GopayPaymentException('Empty or invalid GoPay payment result.');
  }
  if (uc_gopay_setting('debug')) {
    watchdog('uc_gopay', 'GoPay: transaction @id for @order_id. Saved session: <pre>@session</pre> result: <pre>@result</pre>', array(
      '@id' => $returned_payment_session_id,
      '@order_id' => $order_id,
      '@session' => print_r($payment, TRUE),
      '@result' => print_r($result, TRUE)
    ), WATCHDOG_DEBUG, l(t('view'), 'admin/store/orders/' . $order_id));
  }
  
  $original_state = $payment['session_state'];
  
  switch ($result['sessionState']) {
    case GopayHelper::WAITING:
      if ($payment['session_state'] != GopayHelper::WAITING) {
        $payment['session_state'] = $result['sessionState'];
        uc_order_update_status($order_id, 'gopay_pending');
        uc_order_comment_save($order_id, 0, t('Payment is pending at GoPay.'), 'admin');
      }
      break;
    
    case GopayHelper::PAID:
      if (empty($payment['session_state']) || $payment['session_state'] == GopayHelper::WAITING) {
        $payment['session_state'] = $result['sessionState'];
        $amount                   = $payment['total_price'] / 100.00; 
        $comment                  = t('GoPay payment session ID: @sess_id', array(
          '@sess_id' => $returned_payment_session_id
        ));
        uc_payment_enter($order_id, UC_GOPAY_STANDARD_PAYMENT_METHOD, $amount, $order->uid, $payment, $comment);
        uc_cart_complete_sale($order);
        uc_order_comment_save($order_id, 0, t('Payment of @amount @currency submitted through GoPay.', array(
          '@amount' => uc_currency_format($amount, FALSE),
          '@currency' => $payment_currency
        )), 'admin', 'order', 'payment_received');
      }
      break;
    
    case GopayHelper::FAILED:
      if ($payment['session_state'] !== GopayHelper::FAILED) {
        $payment['session_state'] = $result['sessionState'];
        uc_order_comment_save($order_id, 0, t("The customer's attempted payment from a GoPay account failed."), 'admin');
      }
      break;
    
    case GopayHelper::CANCELED:
      if ($payment['session_state'] !== GopayHelper::CANCELED) {
        $payment['session_state'] = $result['sessionState'];
        uc_order_comment_save($order_id, 0, t('GoPay has canceled the reversal and returned !amount !currency to your account.', array(
          '!amount' => uc_currency_format($payment['total_price'], FALSE),
          '!currency' => $payment_currency
        )), 'admin');
      }
      break;
    
    case GopayHelper::TIMEOUTED:
      if ($payment['session_state'] !== GopayHelper::TIMEOUTED) {
        $payment['session_state'] = $result['sessionState'];
        uc_order_comment_save($order_id, 0, t('The authorization of GoPay payment has failed and cannot be captured.'), 'admin');
      }
      break;
    
    default:
      watchdog('uc_gopay', 'GoPay: Unknown payment state.', array(), WATCHDOG_ERROR, l(t('view'), 'admin/store/orders/' . $order_id));
      throw new GopayPaymentException('Unknown GoPay payment state.');
  }
  
  // Save payment status info if changed.
  if ($original_state != $payment['session_state']) {
    watchdog('uc_gopay', 'GoPay: transaction @id for @order_id state changed from %from to %to.', array(
      '@id' => $returned_payment_session_id,
      '@order_id' => $order_id,
      '%from' => $original_state ? $original_state : '--',
      '%to' => $payment['session_state']
    ), WATCHDOG_NOTICE, l(t('view'), 'admin/store/orders/' . $order_id));
    uc_gopay_record_save($payment);
  }
}