<?php

//=============================================================================
//= MENU CALLBACKS
//=============================================================================

/**
 * Successful payment callback.
 */
function uc_gpwebpay_payment_success() {
  uc_gpwebpay_watchlog('uc_gpwebpay', print_r(dbtlog(), 1));
  _uc_gpwebpay_check_response_access();
  
  if (uc_gpwebpay_setting('debug')) {
    watchdog('uc_gpwebpay', 'Access to payment success page. $_REQUEST = %request', array(
      '%request' => print_r($_REQUEST, TRUE)
    ), WATCHDOG_DEBUG);
  }
  
  try {
    $lock_name = 'uc_gpwebpay_' . $_GET['ORDERNUMBER'];
    
    if (!lock_acquire($lock_name, 5)) {
      lock_wait($lock_name);
      lock_acquire($lock_name, 5);
    }
    
    _uc_gpwebpay_complete();
    
    lock_release($lock_name);
    
    // This lets us know it's a legitimate access of the complete page.
    $_SESSION['do_complete'] = TRUE;
    drupal_goto('cart/checkout/complete');
  }
  catch (GpwebpayPaymentException $e) {
    drupal_set_message(t('Error during processing the payment: %error', array(
      '%error' => $e->getMessage()
    )), 'error');
    drupal_goto('cart/checkout/review');
  }
}

/**
 * Failed payment callback.
 */
function uc_gpwebpay_payment_failed() {
  uc_gpwebpay_watchlog('uc_gpwebpay', print_r(dbtlog(), 1));
  
  if (empty($_GET['ORDERNUMBER'])) {
    //drupal_access_denied();
    //exit;
  }
  
  if (uc_gpwebpay_setting('debug')) {
    watchdog('uc_gpwebpay', 'Access to payment failed page. $_REQUEST = %request', array(
      '%request' => print_r($_REQUEST, TRUE)
    ), WATCHDOG_DEBUG);
  }
  unset($_SESSION['cart_order']);
  drupal_set_message(t('Your GPWebPay payment was canceled. Please feel free to continue shopping or contact us for assistance.'));
  drupal_goto('cart');
}

/**
 * Notify callback.
 */
function uc_gpwebpay_soap_notify() {
  uc_gpwebpay_watchlog('uc_gpwebpay', print_r(dbtlog(), 1));
  
  _uc_gpwebpay_check_response_access();
  
  if (uc_gpwebpay_setting('debug')) {
    watchdog('uc_gpwebpay', 'Access to payment notify page. $_REQUEST: <pre>@request</pre>', array(
      '@request' => print_r($_REQUEST, TRUE)
    ), WATCHDOG_DEBUG);
  }
  
  try {
    $lock_name = 'uc_gpwebpay_' . $_GET['ORDERNUMBER'];
    
    if (!lock_acquire($lock_name, 5)) {
      lock_wait($lock_name);
      lock_acquire($lock_name, 5);
    }
    
    _uc_gpwebpay_complete();
    
    lock_release($lock_name);
  }
  catch (GpwebpayPaymentException $e) {
    _uc_gpwebpay_server_error();
  }
}

/**
 * Admin form of the GPWebPay methods.
 */
function uc_gpwebpay_payment_methods_list() {
  uc_gpwebpay_watchlog('uc_gpwebpay', print_r(dbtlog(), 1));
  
  $form = $options = array();
  
  $form['uc_gpwebpay_back_top'] = array(
    '#prefix' => '<p>',
    '#value' => l(t('Back to GPWebPay settings'), 'admin/store/settings/payment/method/gpwebpay_wps/'),
    '#suffix' => '</p>'
  );
  
  $header              = array(
    t('Logo'),
    t('Payment method'),
    t('Code'),
    t('Offline')
  );
  $payment_method_list = uc_gpwebpay_channel_list_all();
  
  if (module_exists('elements')) {
    foreach ($payment_method_list as &$payment_method) {
      $row                            = array();
      $row[]                          = '<img src="' . $payment_method->logo . '">';
      $row[]                          = $payment_method->paymentMethod;
      $row[]                          = $payment_method->code;
      $row[]                          = ($payment_method->offline == 1) ? t('yes') : t('no');
      $options[$payment_method->code] = $row;
    }
    
    $form['uc_gpwebpay_enabled_channels'] = array(
      '#type' => 'tableselect',
      '#header' => $header,
      '#options' => $options,
      '#empty' => t('There are not payment channels available. Server connection error?'),
      '#default_value' => uc_gpwebpay_setting('channels')
    );
  } else {
    foreach ($payment_method_list as &$payment_method) {
      $row                            = sprintf('<img src="%s"> <b>%s</b> <code>(code: %s, offline: %s)</code>', $payment_method->logo, $payment_method->paymentMethod, $payment_method->code, ($payment_method->offline == 1) ? t('yes') : t('no'));
      $options[$payment_method->code] = $row;
    }
    
    if ($options) {
      $form['uc_gpwebpay_enabled_channels'] = array(
        '#type' => 'checkboxes',
        '#options' => $options,
        '#default_value' => uc_gpwebpay_setting('channels')
      );
    } else {
      $form['uc_gpwebpay_enabled_channels_empty'] = array(
        '#type' => 'markup',
        '#prefix' => '<p>',
        '#suffix' => '</p>',
        '#value' => t('There are not payment channels available. Server connection error?')
      );
    }
  }
  
  $form['uc_gpwebpay_back_bottom'] = array(
    '#prefix' => '<p>',
    '#value' => l(t('Back to GPWebPay settings'), 'admin/store/settings/payment/method/gpwebpay_wps/'),
    '#suffix' => '</p>'
  );
  
  return system_settings_form($form);
}


//=============================================================================
//= HELPERS
//=============================================================================

class GpwebpayPaymentException extends Exception {
}

/**
 * Returns 200 OK HTTP header.
 */
function _uc_gpwebpay_server_success() {
  uc_gpwebpay_watchlog('uc_gpwebpay', print_r(dbtlog(), 1));
  if (uc_gpwebpay_setting('debug')) {
    watchdog('uc_gpwebpay', 'Server SUCCESS.');
  }
  header('HTTP/1.1 200 OK');
  exit(0);
}

/**
 * Returns 500 Server error HTTP header.
 */
function _uc_gpwebpay_server_error() {
  uc_gpwebpay_watchlog('uc_gpwebpay', print_r(dbtlog(), 1));
  if (uc_gpwebpay_setting('debug')) {
    watchdog('uc_gpwebpay', 'Server ERROR.');
  }
  header('HTTP/1.1 500 Internal Server Error');
  exit(0);
}

/**
 * Splits variable symbol which came back from GPWebPay to order_id and cart_id.
 *
 * @param string $variable_symbol
 * @return array of order_id [0] and cart_id [1]
 */
function _uc_gpwebpay_parse_variable_symbol($variable_symbol) {
  uc_gpwebpay_watchlog('uc_gpwebpay', print_r(dbtlog(), 1));
  // Parse Order ID and Cart ID
  //$order_id = floor($variable_symbol / 1000000);
  //$cart_id = floor($variable_symbol % 1000000);
  //opravene
  $order_id = floor($variable_symbol / 10000);
  $cart_id = floor($variable_symbol % 10000);
  
  
  // Sanitize order ID and cart ID
  $order_id = intval($order_id);
  $cart_id  = $cart_id ? check_plain($cart_id) : 0;
  return array(
    $order_id,
    $cart_id
  );
}

function _uc_gpwebpay_check_response_access() {
  uc_gpwebpay_watchlog('uc_gpwebpay', print_r(dbtlog(), 1));
  
  $wp_response = new \AdamStipak\Webpay\PaymentResponse((string) @$_REQUEST['OPERATION'], (string) @$_REQUEST['ORDERNUMBER'], (string) @$_REQUEST['MERORDERNUM'], (int) @$_REQUEST['PRCODE'], (int) @$_REQUEST['SRCODE'], (string) @$_REQUEST['RESULTTEXT'], (string) @$_REQUEST['DIGEST'], (string) @$_REQUEST['DIGEST1']);
  
  $payment = uc_gpwebpay_record_find_by_session_id($_REQUEST['ORDERNUMBER']);
  if (!$payment) {
    drupal_access_denied();
    die();
  }
  
  try {
    $webpay = uc_gpwebpay_get_api();
    $webpay->verifyPaymentResponse($wp_response);
  }
  catch (Exception $e) {
    drupal_access_denied();
    exit();
  }
}

/**
 * Process the notification returned from the GPWebPay module to finish the payment
 * process.
 *
 * @throws GpwebpayPaymentException
 */
function _uc_gpwebpay_complete() {
  uc_gpwebpay_watchlog('uc_gpwebpay', print_r(dbtlog(), 1));
  // Assing the values from the GPWebPay request
  $returned_variable_symbol    = $_GET['ORDERNUMBER'];
  $returned_payment_session_id = $_GET['ORDERNUMBER'];
  $returned_merchant           = $_GET['MERCHANTNUMBER'];
  
  // Parse Order ID and Cart ID
  list($order_id, $cart_id) = _uc_gpwebpay_parse_variable_symbol($returned_variable_symbol);
  
  if (!empty($cart_id)) {
    // Needed later by uc_complete_sale to empty the correct cart
    $_SESSION['uc_cart_id'] = $cart_id;
  }
  
  watchdog('uc_gpwebpay', 'Receiving PN at URL for order @order_id. <pre>@debug</pre>', array(
    '@order_id' => $order_id,
    '@debug' => uc_gpwebpay_setting('debug') ? print_r($_GET, TRUE) : ''
  ), WATCHDOG_NOTICE, l(t('view'), 'admin/store/orders/' . $order_id));
  
  $payment = uc_gpwebpay_record_find_by_session_id($returned_payment_session_id);
  if (uc_gpwebpay_setting('debug')) {
    watchdog('uc_gpwebpay', 'Payment session record. <pre>@debug</pre>', array(
      '@debug' => print_r($payment, TRUE)
    ), WATCHDOG_DEBUG, l(t('view'), 'admin/store/orders/' . $order_id));
  }
  
  $order = uc_order_load($order_id);
  if ($order == FALSE) {
    watchdog('uc_gpwebpay', 'GPWebPay response attempted for non-existent order @order_id.', array(
      '@order_id' => $order_id
    ), WATCHDOG_ERROR, l(t('view'), 'admin/store/orders/' . $order_id));
    throw new GpwebpayPaymentException('GPWebPay transaction failed verification');
  } elseif (($order->payment_method != UC_GPWEBPAY_STANDARD_PAYMENT_METHOD) && (!defined('UC_GPWEBPAY_INLINE_STANDARD_PAYMENT_METHOD') || ($order->payment_method != UC_GPWEBPAY_INLINE_STANDARD_PAYMENT_METHOD))) {
    watchdog('uc_gpwebpay', 'Attempt to pay order @order_id using GPWebPay, but pament method %payment_method was selected.', array(
      '@order_id' => $order_id,
      '%payment_method' => $order->payment_method
    ), WATCHDOG_ERROR, l(t('view'), 'admin/store/orders/' . $order_id));
    throw new GpwebpayPaymentException("Missing GPWebPay payment method in the order");
  }
  
  $wp_response = new \AdamStipak\Webpay\PaymentResponse((string) @$_REQUEST['OPERATION'], (string) @$_REQUEST['ORDERNUMBER'], (string) @$_REQUEST['MERORDERNUM'], (int) @$_REQUEST['PRCODE'], (int) @$_REQUEST['SRCODE'], (string) @$_REQUEST['RESULTTEXT'], (string) @$_REQUEST['DIGEST'], (string) @$_REQUEST['DIGEST1']);
  
  try {
    $webpay = uc_gpwebpay_get_api();
    $ok     = $webpay->verifyPaymentResponse($wp_response);

    $payment['result'] = $_REQUEST['RESULTTEXT'] . '|' . $_REQUEST['PRCODE'] . '/' . $_REQUEST['SRCODE'];

    if ($ok) {
      $payment['session_state'] = 'complete';
      $amount                   = $payment['total_price'] / 100.00;
      $comment                  = t('GPWebPay payment session ID: @sess_id', array(
        '@sess_id' => $returned_payment_session_id
      ));
      uc_payment_enter($order_id, UC_GPWEBPAY_STANDARD_PAYMENT_METHOD, $amount, $order->uid, $payment, $comment);
      uc_cart_complete_sale($order);
      uc_order_comment_save($order_id, 0, t('Payment of @amount @currency submitted through GPWebPay.', array(
        '@amount' => uc_currency_format($amount, FALSE),
        '@currency' => $payment_currency
      )), 'admin', 'order', 'pending');

      uc_gpwebpay_record_save($payment);
    } else {
      $payment['session_state'] = 'failed';
      uc_order_comment_save($order_id, 0, t("The customer's attempted payment from a GPWebPay account failed."), 'admin');

      uc_gpwebpay_record_save($payment);
      uc_gpwebpay_payment_failed();
    }
  }
  catch (Exception $e) {
    $payment['session_state'] = 'failed';
    $payment['result'] = $e->getMessage();

    uc_gpwebpay_record_save($payment);
    uc_gpwebpay_payment_failed();
  }

  
}