<?php

/**
 * @file
 * Ubercart Rule integration
 */

define('UC_RULES_MODULE_NAME', 'Ubercart Rules');

/**
 * Implements hook_rules_data_type_info().
 */
function uc_rules_rules_data_type_info() {
  return array(
    'uc_order' => array(
      'label' => t('Order'),
      'class' => 'uc_rules_data_type_order',
      'savable' => TRUE,
      'identifiable' => TRUE,
      'module' => t(UC_RULES_MODULE_NAME),
      'token type' => 'order'
    ),
    'uc_product' => array(
      'label' => t('Product'),
      'class' => 'rules_data_type_node',
      'savable' => TRUE,
      'identifiable' => TRUE,
      'module' => t(UC_RULES_MODULE_NAME),
      'token type' => 'product'
    ),
  );
}

/**
 * Defines the uc_order data type.
 */
class uc_rules_data_type_order extends rules_data_type {

  function save() {
    $order = &$this->get();
    uc_order_save($order);
    return TRUE;
  }

  function load($order_id) {
    return uc_order_load($order_id);
  }

  function get_identifier() {
    $order = &$this->get();
    return $order->order_id;
  }

}

/**
 * Implements hook_rules_event_info().
 */
function uc_rules_rules_event_info() {

  $items['uc_rules_event_order_state_update'] = array(
    'label' => t('Order state is updated'),
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Order'),
      ),
      'old_order_status' => array(
        'type' => 'string',
        'label' => t('Old Order Status'),
      ),
      'new_order_status' => array(
        'type' => 'string',
        'label' => t('New Order Status'),
      ),
    ),
    'module' => t(UC_RULES_MODULE_NAME),
  );

  $items['uc_rules_event_order_state_update_line_item'] = array(
    'label' => t('Order state is updated - per order line'),
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Updated Order'),
      ),
      'uc_line_item' => array(
        'type' => 'uc_line_item',
        'label' => t('Updated Order Line item'),
      ),
      'old_order_status' => array(
        'type' => 'string',
        'label' => t('Old Order Status'),
      ),
      'new_order_status' => array(
        'type' => 'string',
        'label' => t('New Order Status'),
      ),
    ),
    'module' => t(UC_RULES_MODULE_NAME),
  );

  $items['uc_rules_event_order_checkout_complete'] = array(
    'label' => t('Order checkout complete'),
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Updated Order'),
      ),
      'user' => array(
        'type' => 'user',
        'label' => t('User'),
      ),
    ),
    'module' => t(UC_RULES_MODULE_NAME),
  );

  $items['uc_rules_event_payment_entered'] = array(
    'label' => t('Payment entered'),
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Order'),
      ),
      'user' => array(
        'type' => 'user',
        'label' => t('User'),
      ),
    ),
    'module' => t(UC_RULES_MODULE_NAME),
  );

  return $items;
}

/**
 * Implements hook_rules_action_info()
 */
function uc_rules_rules_action_info() {
  $items = array();

  $items['uc_rules_action_load_customer_user'] = array(
    'label' => t('Load user by order'),
    'module' => t(UC_RULES_MODULE_NAME),
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Order'),
      ),
    ),
    'new variables' => array(
      'user_loaded' => array('type' => 'user', 'label' => t('Loaded user')),
    ),
  );

  $items['uc_rules_action_load_product_node_line_item'] = array(
    'label' => t('Load product by order line item'),
    'module' => t(UC_RULES_MODULE_NAME),
    'arguments' => array(
      'uc_line_item' => array(
        'type' => 'uc_line_item',
        'label' => t('Ubercart Order Line Item'),
      ),
    ),
    'new variables' => array(
      'product_loaded' => array('type' => 'uc_product', 'label' => t('Loaded product')),
    ),
  );

  $items['uc_rules_action_load_product'] = array(
    'label' => t('Load a product'),
    'module' => t(UC_RULES_MODULE_NAME),
    'arguments' => array(),
    'new variables' => array(
      'uc_product_loaded' => array('type' => 'uc_product', 'label' => t('Loaded Ubercart product')),
    ),
  );

  $items['uc_rules_action_load_order_by_id'] = array(
    'label' => t('Load order by order ID'),
    'module' => t(UC_RULES_MODULE_NAME),
    'arguments' => array(),
    'eval input' => array('uc_order_id'),
    'new variables' => array(
      'uc_order_loaded' => array('type' => 'uc_order', 'label' => t('Loaded Ubercart Order')),
    ),
  );

  $items['uc_rules_action_send_order_email_invoice'] = array(
    'label' => t('Send invoice email'),
    'module' => t(UC_RULES_MODULE_NAME),
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Order'),
      ),
    ),
    'eval input' => array('from', 'addresses', 'subject'),
  );

  $items['uc_rules_action_order_add_comment'] = array(
    'label' => t('Add a comment to an order'),
    'module' => t(UC_RULES_MODULE_NAME),
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Order'),
      ),
    ),
    'eval input' => array('comment'),
  );

  $items['uc_rules_action_order_update_status'] = array(
    'label' => t('Update order status'),
    'module' => t(UC_RULES_MODULE_NAME),
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Order'),
      ),
    ),
    'eval input' => array('comment'),
  );

  return $items;
}

/**
 * Implements hook_rules_condition_info().
 */
function uc_rules_rules_condition_info() {
  $items = array();

  $items['uc_rules_condition_order_has_status'] = array(
    'label' => t('Check order status'),
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Order'),
      ),
    ),
    'module' => t(UC_RULES_MODULE_NAME),
    'eval input' => array('status'),
  );

  $items['uc_rules_condition_order_has_products'] = array(
    'label' => t("Check order has product(s)"),
    'module' => t(UC_RULES_MODULE_NAME),
    'base' => 'uc_rules_condition_order_has_products',
    'arguments' => array(
      'uc_order' => array(
        'type' => 'uc_order',
        'label' => t('Order'),
      ),
    ),
  );

  if (module_exists('uc_attribute')) {
    $items['uc_rules_condition_products_with_attribute_option'] = array(
      'label' => t('Check order has any product(s) with attribute option'),
      'description' => t('Search the products of an order for a particular option.'),
      'module' => t(UC_RULES_MODULE_NAME),
      'arguments' => array(
        'order' => array(
          'type' => 'uc_order',
          'label' => t('Order')
        ),
      ),
    );
  };

  return $items;
}

/**
 * Updates the order status.
 *
 * @param object $order
 *  Order to change
 * @param array $settings
 *  Settings array including order_status
 *
 * @see uc_rules_action_order_update_status_form()
 */
function uc_rules_action_order_update_status($order, $settings) {
  if (uc_order_update_status($order->order_id, $settings['order_status'])) {
    $order->order_status = $settings['order_status'];
  };
}

/**
 * Adds a new comment to the given order.
 *
 * @param object $order
 *  Order on which the comment should be added
 * @param array $settings
 *
 * @see uc_rules_action_order_add_comment_form()
 */
function uc_rules_action_order_add_comment($order, $settings) {
  uc_order_action_add_comment($order, $settings);
}

/**
 * Sends the invoice e-mail.
 *
 * @param object $order
 * @param array $settings
 *
 * @see uc_rules_action_send_order_email_invoice_form()
 */
function uc_rules_action_send_order_email_invoice($order, $settings) {

  $recipients = array();
  $addresses = $settings['addresses'];

  foreach (explode("\n", $addresses) as $address) {
    $recipients[] = trim($address);
  };

  $settings['message'] = theme('uc_order', $order, $settings['view'], $settings['template']);

  if (empty($recipients)) {
    watchdog('uc_rules', 'Attempted to e-mail an invoice with no recipient.', array(), WATCHDOG_ERROR);
    return;
  };

  foreach ($recipients as $email) {
    $sent = drupal_mail('uc_order', 'action-mail', $email, uc_store_mail_recipient_language($email), $settings, empty($settings['from']) ? uc_store_email_from() : $settings['from']);

    if (!$sent['result']) {
      watchdog('uc_rules', 'Attempt to e-mail invoice for order @order_id to @email failed.', array('@email' => $email, '@order_id' => $order->order_id), WATCHDOG_ERROR);
    };
  };
}

/**
 * Loads a product node associated with the order item line.
 *
 * @param $uc_line_item
 *  Order line Item
 * @param $settings
 *  Settings array
 * @return array
 *  With the loaded node
 */
function uc_rules_action_load_product_node_line_item($uc_line_item, $settings) {
  return array('product_loaded' => node_load($uc_line_item->nid));
}

/**
 * Loads a product from select list or textfield input.
 *
 * @param array $settings
 * @return array
 *
 * @see uc_rules_action_load_product_form()
 */
function uc_rules_action_load_product($settings) {
  $nid = empty($settings['uc_product_id_text']) ? $settings['uc_product_id'] : $settings['uc_product_id_text'];
  $product = node_load($nid);
  return array('uc_product_loaded' => $product);
}

/**
 * Loads the order from an id.
 *
 * @param array $settings
 *  Settings array incl. order id
 * @return array
 *
 * @see uc_rules_action_load_order_by_id_form()
 */
function uc_rules_action_load_order_by_id($settings) {
  $order_id = $settings['uc_order_id'];
  $order = uc_order_load($order_id);
  return array('uc_order_loaded' => $order);
}

/**
 * Loads the user (customer) from the order.
 *
 * @param object $uc_order
 *  Fully loaded Ubercart order.
 * @param array $settings
 * @return array
 */
function uc_rules_action_load_customer_user($uc_order, $settings) {
  $user = user_load(array('uid' => $uc_order->uid));
  return array('user_loaded' => $user);
}

/**
 * Checks if an order has a certain order status.
 *
 * @param object $uc_order
 * @param array $settings
 *  Settings including order status
 * @return bool
 *
 * @see uc_rules_condition_order_has_status_form()
 */
function uc_rules_condition_order_has_status($uc_order, $settings) {
  return ($uc_order->order_status == $settings['status']);
}

/**
 * Checks that the order has the selected combination of products.
 *
 * @param object $order
 * @param array $settings
 * @return bool
 *
 * @see uc_rules_condition_order_has_products_form()
 */
function uc_rules_condition_order_has_products($order, $settings) {
  $order_products = array();
  foreach ($order->products as $product) {
    $order_products[] = $product->nid;
  };
  $required_products = array_intersect($settings['products'], $order_products);
  if ($settings['required']) {
    $required_check = $required_products == $settings['products'];
  }
  else {
    $required_check = (bool) count($required_products);
  };
  if ($settings['forbidden']) {
    $forbidden_products = array_diff($order_products, $settings['products']);
    $forbidden_check = (bool) count($forbidden_products);
  }
  else {
    $forbidden_check = FALSE;
  };
  return $required_check && !$forbidden_check;
}

/**
 * Helper that loads available product options.
 *
 * @return array
 *  Product model keyed by node ID.
 */
function uc_rules_options_product() {
  $options = array();

  $result = db_query('SELECT nid, model FROM {uc_products}');
  while ($product = db_fetch_object($result)) {
    $options[$product->nid] = $product->model;
  };
  asort($options);

  return $options;
}

if (module_exists('uc_attribute')) {

  /**
   * Checks if order has any products with a selection attribute option.
   *
   * @param object $order
   * @param int $oid
   * @return bool
   *
   * @see uc_rules_condition_products_with_attribute_option_form()
   */
  function uc_rules_condition_products_with_attribute_option($order, $oid) {
    $option = uc_attribute_option_load($oid);
    $attribute = uc_attribute_load($option->aid);

    foreach ($order->products as $product) {
      if (!isset($product->data['attributes'])) {
        continue;
      };

      $attributes = $product->data['attributes'];

      // Once the order is made, the attribute data is changed to just the names.
      // If we can't find it by ID, check the names.
      if (is_int(key($attributes))) {
        if (isset($attributes[$oid])) {
          return TRUE;
        };
      }
      elseif (isset($attributes[$attribute->name][$oid])) {
        return TRUE;
      };
    };

    return FALSE;
  }
}